package football;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;

/**
 * Classe équipe, contient des joueurs
 * @author Adrien Poupa
 */
public class Equipe {
    public static final int NOMBREJOUEURS = 11;
    private String club;
    ArrayList<Joueur> container;

    /**
     * Création d'une équipe à partir d'un nom de club
     * @param club nom du club
     */
    public Equipe(String club) {
        this.club = club;
        this.container = new ArrayList<Joueur>();
    }


    /**
     * Ajout d'un joueur dans le container de l'équipe
     * @param joueur joueur à jouter
     * @throws EquipePleineException si l'équipe est pleine
     * @throws JoueurHorsClubEquipeException si on ajoute un joueur qui n'est pas dans l'équipe
     */
    public void ajout(Joueur joueur)
        throws EquipePleineException, JoueurHorsClubEquipeException
    {
        if (container.size() == NOMBREJOUEURS) {
            throw new EquipePleineException("Les 12 apôtres ne jouaient pas au football!");
        }

        if (!joueur.getClub().equals(club)) {
            throw new JoueurHorsClubEquipeException(joueur.getClub(), club);
        }

        joueur.equipe = this; // Problème possible?
        container.add(joueur);
    }

    /**
     * Tri des joueurs de l'équipe par salaire
     */
    public void triSalaire() {
        Collections.sort(container, new Comparator<Joueur>() {
            @Override
            public int compare(Joueur a, Joueur b) {
                return Double.compare(a.getSalaire(), b.getSalaire());
            }
        });
    }

    /**
     * Affichage de l'équipe
     * @return Equipe à afficher
     */
    @Override
    public String toString() {
        StringBuilder returnString = new StringBuilder();

        for (Joueur joueur : container) {
            returnString.append(joueur.toString());
            returnString.append(System.getProperty("line.separator"));
        }

        return "Equipe{" +
                "club='" + club + '\'' +
                ", joueurs=" + System.getProperty("line.separator") + returnString.toString() +
                '}';
    }

    /**
     * Surcharge de equals
     * @param o objet à comparer
     * @return true/false
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Equipe equipe = (Equipe) o;

        if (!club.equals(equipe.club)) return false;
        return container != null ? container.equals(equipe.container) : equipe.container == null;

    }

    /**
     * Surcharge de hashCode
     * @return hash int unique
     */
    @Override
    public int hashCode() {
        int result = club.hashCode();
        result = 31 * result + (container != null ? container.hashCode() : 0);
        return result;
    }
}
